```sql
-- Database schema for EduSphere platform

-- Create database
CREATE DATABASE IF NOT EXISTS edusphere;
USE edusphere;

-- Users table
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    role ENUM('student', 'instructor', 'admin') NOT NULL DEFAULT 'student',
    avatar VARCHAR(255),
    bio TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Courses table
CREATE TABLE courses (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    category_id INT,
    instructor_id INT NOT NULL,
    price DECIMAL(10,2) DEFAULT 0,
    image VARCHAR(255),
    rating DECIMAL(3,2) DEFAULT 0,
    duration VARCHAR(50),
    is_featured BOOLEAN DEFAULT FALSE,
    is_published BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (instructor_id) REFERENCES users(id)
);

-- Categories table
CREATE TABLE categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    icon VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Course sections table
CREATE TABLE course_sections (
    id INT AUTO_INCREMENT PRIMARY KEY,
    course_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    position INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE
);

-- Lessons table
CREATE TABLE lessons (
    id INT AUTO_INCREMENT PRIMARY KEY,
    section_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    content TEXT,
    video_url VARCHAR(255),
    duration VARCHAR(50),
    position INT NOT NULL,
    is_free BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (section_id) REFERENCES course_sections(id) ON DELETE CASCADE
);

-- Enrollments table
CREATE TABLE enrollments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    course_id INT NOT NULL,
    progress INT DEFAULT 0,
    completed BOOLEAN DEFAULT FALSE,
    enrolled_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (course_id) REFERENCES courses(id)
);

-- Reviews table
CREATE TABLE reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    course_id INT NOT NULL,
    rating INT NOT NULL,
    comment TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (course_id) REFERENCES courses(id)
);

-- Media resources table
CREATE TABLE media_resources (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    type ENUM('video', 'podcast', 'article', 'book') NOT NULL,
    file_url VARCHAR(255) NOT NULL,
    thumbnail_url VARCHAR(255),
    duration VARCHAR(50),
    author_id INT,
    published_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (author_id) REFERENCES users(id)
);

-- Insert sample data
INSERT INTO users (name, email, password, role, avatar, bio) VALUES
('أحمد محمد', 'ahmed@example.com', '$2a$10$examplehashedpassword', 'student', 'http://static.photos/people/200x200/1', NULL),
('سارة عبدالله', 'sara@example.com', '$2a$10$examplehashedpassword', 'student', 'http://static.photos/people/200x200/2', NULL),
('د. علي حسين', 'ali@example.com', '$2a$10$examplehashedpassword', 'instructor', 'http://static.photos/people/200x200/3', 'أستاذ علوم الحاسوب بجامعة الملك سعود');

INSERT INTO categories (name, icon) VALUES
('برمجة', 'code'),
('تصميم', 'palette'),
('إدارة', 'briefcase'),
('أعمال', 'dollar-sign'),
('تسويق', 'trending-up');

INSERT INTO courses (title, description, category_id, instructor_id, price, image, rating, duration) VALUES
('تعلم البرمجة بلغة Python', 'دورة شاملة لتعلم أساسيات البرمجة بلغة Python من الصفر حتى الاحتراف.', 1, 3, 0, 'http://static.photos/technology/640x360/1', 4.8, '12 ساعة'),
('أساسيات التصميم الجرافيكي', 'تعلم أساسيات التصميم الجرافيكي باستخدام أدوبي فوتوشوب وإليستريتور.', 2, 3, 150, 'http://static.photos/design/640x360/1', 4.5, '8 ساعات');

-- Create indexes for better performance
CREATE INDEX idx_courses_category ON courses(category_id);
CREATE INDEX idx_enrollments_user ON enrollments(user_id);
CREATE INDEX idx_enrollments_course ON enrollments(course_id);
CREATE INDEX idx_reviews_course ON reviews(course_id);
```
___METADATA_START___
{"repoId":"Mahmoud646/edusphere","isNew":false,"userName":"Mahmoud646"}
___METADATA_END___