```sql
-- Database Schema for EduSphere Platform
-- Tables creation with Arabic comments where appropriate

-- Users Table (طلاب، مدرسون، مدراء)
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL COMMENT 'الاسم الكامل',
    email VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL COMMENT 'كلمة المرور مشفرة',
    role ENUM('student', 'instructor', 'admin') NOT NULL DEFAULT 'student' COMMENT 'دور المستخدم',
    avatar VARCHAR(255) COMMENT 'صورة المستخدم',
    bio TEXT COMMENT 'السيرة الذاتية للمدرسين',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'تاريخ التسجيل',
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Categories Table (التصنيفات)
CREATE TABLE categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) NOT NULL COMMENT 'اسم التصنيف',
    icon VARCHAR(50) COMMENT 'أيقونة التصنيف',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Courses Table (الدورات)
CREATE TABLE courses (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL COMMENT 'عنوان الدورة',
    description TEXT COMMENT 'وصف الدورة',
    category_id INT NOT NULL COMMENT 'التصنيف',
    instructor_id INT NOT NULL COMMENT 'المدرب',
    price DECIMAL(10,2) DEFAULT 0 COMMENT 'السعر',
    image VARCHAR(255) COMMENT 'صورة الدورة',
    duration VARCHAR(50) COMMENT 'مدة الدورة',
    level ENUM('beginner', 'intermediate', 'advanced') DEFAULT 'beginner' COMMENT 'مستوى الصعوبة',
    is_featured BOOLEAN DEFAULT FALSE COMMENT 'دورة مميزة',
    status ENUM('draft', 'published', 'archived') DEFAULT 'draft' COMMENT 'حالة الدورة',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id),
    FOREIGN KEY (instructor_id) REFERENCES users(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Sections Table (أقسام الدورة)
CREATE TABLE sections (
    id INT PRIMARY KEY AUTO_INCREMENT,
    course_id INT NOT NULL,
    title VARCHAR(255) NOT NULL COMMENT 'عنوان القسم',
    order_num INT NOT NULL DEFAULT 0 COMMENT 'ترتيب القسم',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Lessons Table (الدروس)
CREATE TABLE lessons (
    id INT PRIMARY KEY AUTO_INCREMENT,
    section_id INT NOT NULL,
    title VARCHAR(255) NOT NULL COMMENT 'عنوان الدرس',
    content TEXT COMMENT 'محتوى الدرس (للدروس النصية)',
    video_url VARCHAR(255) COMMENT 'رابط الفيديو',
    duration INT COMMENT 'مدة الدرس بالدقائق',
    order_num INT NOT NULL DEFAULT 0 COMMENT 'ترتيب الدرس',
    is_free BOOLEAN DEFAULT FALSE COMMENT 'هل الدرس مجاني؟',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (section_id) REFERENCES sections(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Enrollments Table (تسجيلات الدورات)
CREATE TABLE enrollments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    course_id INT NOT NULL,
    progress INT DEFAULT 0 COMMENT 'النسبة المئوية للإنجاز',
    completed BOOLEAN DEFAULT FALSE COMMENT 'هل اكتملت الدورة؟',
    enrolled_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL COMMENT 'تاريخ الإنجاز',
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (course_id) REFERENCES courses(id),
    UNIQUE KEY unique_enrollment (user_id, course_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Reviews Table (التقييمات)
CREATE TABLE reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    course_id INT NOT NULL,
    rating TINYINT NOT NULL CHECK (rating BETWEEN 1 AND 5) COMMENT 'التقييم من 1 إلى 5',
    comment TEXT COMMENT 'تعليق الطالب',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (course_id) REFERENCES courses(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Media Resources Table (الوسائط التعليمية)
CREATE TABLE media_resources (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    type ENUM('video', 'podcast', 'article', 'ebook') NOT NULL,
    category_id INT,
    file_url VARCHAR(255) NOT NULL,
    thumbnail_url VARCHAR(255),
    duration VARCHAR(50) COMMENT 'مدة الفيديو/البودكاست',
    author VARCHAR(100),
    views INT DEFAULT 0 COMMENT 'عدد المشاهدات',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Sample Data Insertion (إدخال بيانات نموذجية)
-- Users
INSERT INTO users (name, email, password, role, avatar, bio, created_at) VALUES
('أحمد محمد', 'ahmed@example.com', '$2a$10$examplehashedpassword', 'student', 'http://static.photos/people/200x200/1', NULL, '2023-01-15 10:30:00'),
('سارة عبدالله', 'sara@example.com', '$2a$10$examplehashedpassword', 'student', 'http://static.photos/people/200x200/2', NULL, '2023-02-20 14:45:00'),
('د. علي حسين', 'ali@example.com', '$2a$10$examplehashedpassword', 'instructor', 'http://static.photos/people/200x200/3', 'أستاذ علوم الحاسوب بجامعة الملك سعود', '2022-11-10 08:15:00'),
('مدير النظام', 'admin@edusphere.com', '$2a$10$examplehashedpassword', 'admin', 'http://static.photos/people/200x200/4', NULL, '2022-10-01 09:00:00');

-- Categories
INSERT INTO categories (name, icon, created_at) VALUES
('برمجة', 'code', NOW()),
('تصميم', 'palette', NOW()),
('إدارة', 'briefcase', NOW()),
('أعمال', 'dollar-sign', NOW()),
('تسويق', 'trending-up', NOW());

-- Courses
INSERT INTO courses (title, description, category_id, instructor_id, price, image, duration, level, is_featured, status, created_at) VALUES
('تعلم البرمجة بلغة Python', 'دورة شاملة لتعلم أساسيات البرمجة بلغة Python من الصفر حتى الاحتراف.', 1, 3, 0, 'http://static.photos/technology/640x360/1', '12 ساعة', 'beginner', TRUE, 'published', '2023-01-05 09:00:00'),
('أساسيات التصميم الجرافيكي', 'تعلم أساسيات التصميم الجرافيكي باستخدام أدوبي فوتوشوب وإليستريتور.', 2, 3, 150, 'http://static.photos/design/640x360/1', '8 ساعات', 'beginner', TRUE, 'published', '2023-02-10 11:30:00'),
('إدارة المشاريع الاحترافية', 'تعلم منهجيات إدارة المشاريع مثل Agile و Scrum لإدارة المشاريع بفعالية.', 3, 3, 250, 'http://static.photos/office/640x360/1', '10 ساعات', 'intermediate', FALSE, 'published', '2023-03-15 14:00:00');

-- Sections and Lessons for first course
INSERT INTO sections (course_id, title, order_num, created_at) VALUES
(1, 'المقدمة', 1, NOW()),
(1, 'أساسيات Python', 2, NOW());

INSERT INTO lessons (section_id, title, content, video_url, duration, order_num, is_free, created_at) VALUES
(1, 'مقدمة عن Python', 'تعريف لغة Python واستخداماتها', 'https://youtube.com/embed/lesson1', 30, 1, TRUE, NOW()),
(1, 'تثبيت الأدوات', 'كيفية تثبيت Python والمحرر', NULL, 20, 2, TRUE, NOW()),
(2, 'المتغيرات وأنواع البيانات', 'تعلم كيفية تعريف المتغيرات في Python', 'https://youtube.com/embed/lesson2', 45, 1, FALSE, NOW());

-- Enrollments
INSERT INTO enrollments (user_id, course_id, progress, completed, enrolled_at) VALUES
(1, 1, 35, FALSE, '2023-03-01 10:00:00'),
(2, 2, 75, FALSE, '2023-03-05 14:30:00'),
(1, 3, 0, FALSE, '2023-04-10 09:15:00');

-- Reviews
INSERT INTO reviews (user_id, course_id, rating, comment, created_at) VALUES
(1, 1, 5, 'دورة ممتازة ومحتوى غني', '2023-03-10 16:45:00'),
(2, 2, 4, 'شرح واضح ولكن يحتاج المزيد من الأمثلة', '2023-03-15 11:20:00');

-- Media Resources
INSERT INTO media_resources (title, description, type, category_id, file_url, thumbnail_url, duration, author, views, created_at) VALUES
('أساسيات البرمجة للمبتدئين', 'دورة تمهيدية لتعلم أساسيات البرمجة بلغة Python للمبتدئين بدون أي خبرة سابقة', 'video', 1, 'https://youtube.com/embed/video1', 'http://static.photos/education/640x360/10', '15 دقيقة', 'د. علي حسين', 1250, NOW()),
('مستقبل الذكاء الاصطناعي', 'حوار مع د. أحمد سليمان حول آخر تطورات الذكاء الاصطناعي وتأثيره على سوق العمل', 'podcast', 1, 'https://podcast.com/episode1', 'http://static.photos/technology/640x360/11', '45 دقيقة', 'د. أحمد سليمان', 850, NOW()),
('10 نصائح للتعلم الفعال', 'أساليب علمية مثبتة لتحسين قدرتك على التعلم والاستيعاب بشكل أسرع', 'article', NULL, 'https://edusphere.com/articles/10-tips', 'http://static.photos/education/640x360/12', NULL, 'د. ليلى أحمد', 3200, NOW());
```
___METADATA_START___
{"repoId":"Mahmoud646/edusphere","isNew":false,"userName":"Mahmoud646"}
___METADATA_END___