
// EduSphere Main Application Script - Shared across all pages
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all tooltips
    initTooltips();
    
    // Load courses data
    loadCourses();
    
    // Setup event listeners
    setupEventListeners();
    
    // Initialize mobile menu toggle
    initMobileMenu();
});

/**
 * Initialize tooltips using Feather Icons
 */
function initTooltips() {
    const tooltipElements = document.querySelectorAll('[data-tooltip]');
    tooltipElements.forEach(el => {
        const tooltipText = el.getAttribute('data-tooltip');
        el.innerHTML += `<span class="tooltip-text">${tooltipText}</span>`;
    });
}
/**
 * Load courses from local db.json
 */
/**
 * Load courses from API or fallback to local data
 */
async function loadCourses() {
try {
        const response = await fetch('db.json');
        const data = await response.json();
        renderCourses(data.courses);
    } catch (error) {
console.error('Failed to load courses:', error);
        console.error('Failed to load courses:', error);
        // Fallback to hardcoded data if db.json fails
        renderCourses([
            {
                id: 1,
                title: "تعلم البرمجة بلغة Python",
                category: "برمجة",
                instructor: "د. علي حسين",
                price: "مجاناً",
                image: "http://static.photos/technology/640x360/1",
                rating: 4.8,
                students: 1250,
                duration: "12 ساعة",
                description: "دورة شاملة لتعلم أساسيات البرمجة بلغة Python من الصفر حتى الاحتراف."
            },
            {
                id: 2,
                title: "أساسيات التصميم الجرافيكي",
                category: "تصميم",
                instructor: "أ. سامية خالد",
                price: "150 ر.س",
                image: "http://static.photos/design/640x360/1",
                rating: 4.5,
                students: 850,
                duration: "8 ساعات",
                description: "تعلم أساسيات التصميم الجرافيكي باستخدام أدوبي فوتوشوب وإليستريتور."
            }
        ]);
}
}
/**
 * Render courses to the DOM
 */
/**
 * Render courses to the DOM with consistent styling
 */
function renderCourses(courses) {
const container = document.getElementById('courses-container') || 
                      document.getElementById('popular-courses');
    
    if (!container) return;
    
    container.innerHTML = courses.map(course => `
        <div class="bg-white rounded-xl overflow-hidden shadow-md hover:shadow-lg transition duration-300">
            <img src="${course.image}" alt="${course.title}" class="w-full h-48 object-cover">
            <div class="p-6">
                <div class="flex justify-between items-start mb-2">
                    <h3 class="font-semibold text-lg">${course.title}</h3>
                    <span class="bg-indigo-100 text-indigo-800 text-xs px-2 py-1 rounded">${course.category}</span>
                </div>
                <p class="text-gray-600 text-sm mb-4">${course.instructor}</p>
                <div class="flex justify-between items-center">
                    <span class="font-medium text-indigo-600">${course.price === 0 ? 'مجاناً' : course.price + ' ر.س'}</span>
                    <a href="course.html?id=${course.id}" class="text-indigo-600 hover:text-indigo-800 text-sm font-medium">عرض التفاصيل</a>
                </div>
            </div>
        </div>
    `).join('');
}
/**
 * Render star rating
 */
function renderStars(rating) {
    const fullStars = Math.floor(rating);
    const hasHalfStar = rating % 1 >= 0.5;
    let stars = '';
    
    for (let i = 0; i < 5; i++) {
        if (i < fullStars) {
            stars += '<i data-feather="star" class="filled"></i>';
        } else if (i === fullStars && hasHalfStar) {
            stars += '<i data-feather="star" class="half"></i>';
        } else {
            stars += '<i data-feather="star" class="empty"></i>';
        }
    }
    
    return stars;
}

/**
 * Setup event listeners for interactive elements
 */
/**
 * Setup all common event listeners for navigation/interactions
 */
function setupEventListeners() {
// Search functionality
    const searchForm = document.getElementById('search-form');
    if (searchForm) {
        searchForm.addEventListener('submit', function(e) {
            e.preventDefault();
            const query = this.querySelector('input').value;
            searchCourses(query);
        });
    }
    
    // Filter buttons
    const filterButtons = document.querySelectorAll('.filter-btn');
    filterButtons.forEach(btn => {
        btn.addEventListener('click', function() {
            const category = this.dataset.category;
            filterCourses(category);
        });
    });
}

/**
 * Search courses by query
 */
/**
 * Search courses by query from db.json
 */
async function searchCourses(query) {
    try {
        const response = await fetch('db.json');
        const data = await response.json();
        const results = data.courses.filter(course => 
            course.title.includes(query) || 
            course.description.includes(query) ||
            course.category.includes(query)
        );
        renderCourses(results);
    } catch (error) {
        console.error('Search failed:', error);
    }
}
/**
 * Filter courses by category
 */
/**
 * Filter courses by category from db.json
 */
async function filterCourses(category) {
    try {
        const response = await fetch('db.json');
        const data = await response.json();
        const results = category === 'all' 
            ? data.courses 
            : data.courses.filter(course => course.category === category);
        renderCourses(results);
    } catch (error) {
        console.error('Filter failed:', error);
    }
}
/**
 * Initialize mobile menu toggle
 */
function initMobileMenu() {
    const menuToggle = document.getElementById('mobile-menu-toggle');
    const mobileMenu = document.getElementById('mobile-menu');
    
    if (menuToggle && mobileMenu) {
        menuToggle.addEventListener('click', function() {
            mobileMenu.classList.toggle('hidden');
            this.querySelector('svg').setAttribute('data-feather', 
                mobileMenu.classList.contains('hidden') ? 'menu' : 'x');
            feather.replace();
        });
    }
}
// User authentication functions
function authenticateUser(email, password, role) {
    const users = JSON.parse(localStorage.getItem('users')) || [];
    return users.find(u => 
        u.email === email && 
        u.password === password && 
        u.role === role
    );
}
// Get course details by ID
async function getCourseDetails(id) {
    try {
        const response = await fetch('db.json');
        const data = await response.json();
        return data.courses.find(c => c.id === parseInt(id));
    } catch (error) {
        console.error('Failed to get course details:', error);
        return null;
    }
}

// Get user enrolled courses
async function getUserCourses(userId) {
    try {
        const response = await fetch('db.json');
        const data = await response.json();
        const enrollments = data.enrollments.filter(e => e.userId === userId);
        return data.courses.filter(c => 
            enrollments.some(e => e.courseId === c.id)
        );
    } catch (error) {
        console.error('Failed to get user courses:', error);
        return [];
    }
}
